import { useState, useEffect } from 'react'
import { motion, useScroll, useTransform } from 'framer-motion'
import { Calendar, Star, User, Navigation, Clock, TrendingUp, Award, Compass, Route } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyTours.css'

const MyTours = () => {
  const [tours, setTours] = useState([])
  const [selectedTour, setSelectedTour] = useState(null)
  const { scrollY } = useScroll()
  const y = useTransform(scrollY, [0, 300], [0, -50])

  useEffect(() => {
    const saved = localStorage.getItem('excursionsTours')
    if (saved) {
      try {
        setTours(JSON.parse(saved))
      } catch (e) {
        setTours([])
      }
    } else {
      const defaultTours = [
        {
          id: 1,
          route: 'Тайны Красной площади',
          guide: 'Анна Петрова',
          date: '2025-04-15',
          rating: 5,
          duration: '3 часа',
          theme: 'Историческая',
          highlights: ['Кремль', 'Собор Василия Блаженного', 'ГУМ']
        },
        {
          id: 2,
          route: 'Гастрономические секреты Замоскворечья',
          guide: 'Михаил Соколов',
          date: '2025-04-12',
          rating: 4,
          duration: '2.5 часа',
          theme: 'Гастрономическая',
          highlights: ['Традиционные блюда', 'Исторические кафе', 'Дегустации']
        },
        {
          id: 3,
          route: 'Природные сокровища Коломенского',
          guide: 'Елена Волкова',
          date: '2025-04-10',
          rating: 5,
          duration: '4 часа',
          theme: 'Природная',
          highlights: ['Древние дубы', 'Яблоневые сады', 'Речные пейзажи']
        }
      ]
      setTours(defaultTours)
      localStorage.setItem('excursionsTours', JSON.stringify(defaultTours))
    }
  }, [])

  const totalTours = tours.length
  const averageRating = tours.length > 0
    ? (tours.reduce((sum, t) => sum + t.rating, 0) / tours.length).toFixed(1)
    : 0
  const upcomingTours = tours.filter(t => new Date(t.date) >= new Date()).length
  const completedTours = tours.filter(t => new Date(t.date) < new Date()).length

  return (
    <div className="my-tours">
      <div className="container">
        <motion.section 
          className="hero-immersive"
          style={{ y }}
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.8 }}
        >
          <div className="hero-content-wrapper">
            <motion.div 
              className="hero-badge-immersive"
              initial={{ scale: 0, rotate: -180 }}
              animate={{ scale: 1, rotate: 0 }}
              transition={{ delay: 0.2, type: "spring", stiffness: 200 }}
            >
              <Compass size={20} strokeWidth={2.5} />
              <span>ИССЛЕДУЙТЕ ГОРОД</span>
            </motion.div>
            <h1 className="hero-title-immersive">
              Мои <span className="title-highlight">экскурсии</span>
            </h1>
            <p className="hero-description-immersive">
              Откройте для себя удивительные маршруты и погрузитесь в историю, 
              культуру и природу нашего города вместе с опытными гидами
            </p>
          </div>
        </motion.section>

        <motion.section 
          className="stats-immersive"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8, delay: 0.3 }}
        >
          <div className="stats-timeline">
            <motion.div 
              className="stat-timeline-item"
              whileHover={{ scale: 1.05, x: 10 }}
              transition={{ type: "spring", stiffness: 300 }}
            >
              <div className="stat-timeline-dot" style={{ background: 'var(--gradient-1)' }}>
                <Navigation size={24} strokeWidth={2} />
              </div>
              <div className="stat-timeline-content">
                <div className="stat-timeline-value">{totalTours}</div>
                <div className="stat-timeline-label">Всего экскурсий</div>
              </div>
            </motion.div>
            <motion.div 
              className="stat-timeline-item"
              whileHover={{ scale: 1.05, x: 10 }}
              transition={{ type: "spring", stiffness: 300 }}
            >
              <div className="stat-timeline-dot" style={{ background: 'var(--gradient-2)' }}>
                <Star size={24} strokeWidth={2} />
              </div>
              <div className="stat-timeline-content">
                <div className="stat-timeline-value">{averageRating}</div>
                <div className="stat-timeline-label">Средний рейтинг</div>
              </div>
            </motion.div>
            <motion.div 
              className="stat-timeline-item"
              whileHover={{ scale: 1.05, x: 10 }}
              transition={{ type: "spring", stiffness: 300 }}
            >
              <div className="stat-timeline-dot" style={{ background: 'var(--gradient-3)' }}>
                <Calendar size={24} strokeWidth={2} />
              </div>
              <div className="stat-timeline-content">
                <div className="stat-timeline-value">{upcomingTours}</div>
                <div className="stat-timeline-label">Предстоящих</div>
              </div>
            </motion.div>
            <motion.div 
              className="stat-timeline-item"
              whileHover={{ scale: 1.05, x: 10 }}
              transition={{ type: "spring", stiffness: 300 }}
            >
              <div className="stat-timeline-dot" style={{ background: 'var(--gradient-1)' }}>
                <Award size={24} strokeWidth={2} />
              </div>
              <div className="stat-timeline-content">
                <div className="stat-timeline-value">{completedTours}</div>
                <div className="stat-timeline-label">Завершено</div>
              </div>
            </motion.div>
          </div>
        </motion.section>

        <ImageSlider />

        <motion.section 
          className="tours-immersive"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8, delay: 0.4 }}
        >
          <div className="section-header-immersive">
            <div className="section-line"></div>
            <h2 className="section-title-immersive">Мои экскурсии</h2>
            <div className="section-line"></div>
          </div>
          {tours.length === 0 ? (
            <div className="empty-immersive">
              <div className="empty-compass">
                <Compass size={64} strokeWidth={1.5} />
                <div className="compass-ring"></div>
              </div>
              <h3 className="empty-title-immersive">Начните свое путешествие</h3>
              <p className="empty-text-immersive">Запишитесь на первую экскурсию и откройте для себя новые горизонты</p>
            </div>
          ) : (
            <div className="tours-map-layout">
              {tours.map((tour, index) => (
                <motion.div
                  key={tour.id}
                  className={`tour-map-card ${selectedTour === tour.id ? 'expanded' : ''}`}
                  initial={{ opacity: 0, scale: 0.9 }}
                  animate={{ opacity: 1, scale: 1 }}
                  transition={{ delay: index * 0.15 }}
                  whileHover={{ scale: 1.02, zIndex: 10 }}
                  onClick={() => setSelectedTour(selectedTour === tour.id ? null : tour.id)}
                >
                  <div className="tour-map-content">
                    <div className="tour-map-header">
                      <div className="tour-theme-badge" style={{ background: 'var(--gradient-1)' }}>
                        {tour.theme}
                      </div>
                      <div className="tour-rating-compact">
                        {Array.from({ length: 5 }).map((_, i) => (
                          <Star
                            key={i}
                            size={16}
                            strokeWidth={2}
                            fill={i < tour.rating ? '#F18F01' : 'none'}
                            color={i < tour.rating ? '#F18F01' : '#D4D4D4'}
                          />
                        ))}
                      </div>
                    </div>
                    <h3 className="tour-route-map">{tour.route}</h3>
                    <div className="tour-info-grid">
                      <div className="tour-info-item">
                        <User size={18} strokeWidth={2} />
                        <span>{tour.guide}</span>
                      </div>
                      <div className="tour-info-item">
                        <Calendar size={18} strokeWidth={2} />
                        <span>{new Date(tour.date).toLocaleDateString('ru-RU', { 
                          day: 'numeric', 
                          month: 'long' 
                        })}</span>
                      </div>
                      <div className="tour-info-item">
                        <Clock size={18} strokeWidth={2} />
                        <span>{tour.duration}</span>
                      </div>
                    </div>
                    {selectedTour === tour.id && (
                      <motion.div 
                        className="tour-highlights"
                        initial={{ opacity: 0, height: 0 }}
                        animate={{ opacity: 1, height: 'auto' }}
                        transition={{ duration: 0.3 }}
                      >
                        <div className="highlights-title">
                          <Route size={18} strokeWidth={2} />
                          <span>Ключевые точки</span>
                        </div>
                        <div className="highlights-list">
                          {tour.highlights?.map((highlight, i) => (
                            <div key={i} className="highlight-item">{highlight}</div>
                          ))}
                        </div>
                      </motion.div>
                    )}
                  </div>
                </motion.div>
              ))}
            </div>
          )}
        </motion.section>

        <motion.section 
          className="insights-section"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8, delay: 0.5 }}
        >
          <div className="section-header-immersive">
            <div className="section-line"></div>
            <h2 className="section-title-immersive">Интересные факты</h2>
            <div className="section-line"></div>
          </div>
          <div className="insights-grid">
            <motion.div 
              className="insight-card"
              whileHover={{ rotate: 2, scale: 1.05 }}
              transition={{ type: "spring", stiffness: 300 }}
            >
              <div className="insight-number">01</div>
              <h3 className="insight-title">Исторические тайны</h3>
              <p className="insight-text">
                Каждая улица хранит удивительные истории. Наши гиды раскроют секреты, 
                которые не найти в обычных путеводителях.
              </p>
            </motion.div>
            <motion.div 
              className="insight-card"
              whileHover={{ rotate: -2, scale: 1.05 }}
              transition={{ type: "spring", stiffness: 300 }}
            >
              <div className="insight-number">02</div>
              <h3 className="insight-title">Кулинарные традиции</h3>
              <p className="insight-text">
                Погрузитесь в мир местной кухни. Узнайте рецепты, передаваемые из поколения 
                в поколение, и попробуйте блюда, которые готовят только здесь.
              </p>
            </motion.div>
            <motion.div 
              className="insight-card"
              whileHover={{ rotate: 2, scale: 1.05 }}
              transition={{ type: "spring", stiffness: 300 }}
            >
              <div className="insight-number">03</div>
              <h3 className="insight-title">Природные чудеса</h3>
              <p className="insight-text">
                Откройте для себя уникальные природные уголки прямо в черте города. 
                Древние парки, заповедные зоны и скрытые сады ждут вас.
              </p>
            </motion.div>
            <motion.div 
              className="insight-card"
              whileHover={{ rotate: -2, scale: 1.05 }}
              transition={{ type: "spring", stiffness: 300 }}
            >
              <div className="insight-number">04</div>
              <h3 className="insight-title">Архитектурные шедевры</h3>
              <p className="insight-text">
                Исследуйте здания, которые рассказывают историю целых эпох. 
                От древних храмов до современных небоскребов.
              </p>
            </motion.div>
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default MyTours
